# 機能設計書 43-Git HEAD取得

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるGit HEAD取得機能の設計を記載する。この機能は、Gitリポジトリの現在のコミットハッシュ（HEAD参照）を取得する機能を提供する。

### 本機能の処理概要

Git HEAD取得機能は、ソースコードディレクトリ内のGitリポジトリから現在チェックアウトされているコミットのハッシュ値を取得する。ブランチを参照している場合とdetached HEAD状態の両方に対応し、ソースコードの一意識別子として使用される。

**業務上の目的・背景**：パッケージビルドシステムでは、ビルドに使用したソースコードのバージョンを追跡する必要がある。Gitリポジトリからソースを取得する場合、現在のコミットハッシュがソースの識別子となる。この情報はビルドの再現性確保やデバッグ時のバージョン特定に重要である。

**機能の利用シーン**：
- ソースフェッチ後のコミットハッシュ記録
- ビルドメタデータ（source_info.toml）への書き込み
- ソース更新の必要性判定
- リビルド判定のためのバージョン比較

**主要な処理内容**：
1. .git/HEADファイルを読み込む
2. 参照形式（ref: refs/heads/...）かdetached形式かを判定する
3. 参照形式の場合、参照先ファイルまたはpacked-refsから実際のコミットハッシュを取得する
4. コミットハッシュとdetached状態フラグを返却する

**関連システム・外部連携**：
- Gitリポジトリ（.gitディレクトリ構造）

**権限による制御**：特になし（ファイル読み取り権限のみ必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連せず、内部ユーティリティとして使用 |

## 機能種別

ユーティリティ / Git操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| dir | &PathBuf | Yes | Gitリポジトリのルートディレクトリパス | .git/HEADファイルが存在すること |

### 入力データソース

- ローカルファイルシステム上のGitリポジトリ
  - .git/HEAD
  - .git/refs/heads/{branch}
  - .git/packed-refs

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| コミットハッシュ | String | 40文字のSHA-1ハッシュ（16進数） |
| detached状態 | bool | trueならdetached HEAD、falseならブランチ参照 |

### 出力先

- 関数の戻り値としてResult<(String, bool), String>で返却

## 処理フロー

### 処理シーケンス

```
1. get_git_head_rev関数呼び出し
   └─ ディレクトリパスを受け取る
2. .git/HEADファイルの読み込み
   └─ fs::read_to_string()
3. HEAD内容の形式判定
   └─ "ref: "で始まるかどうか
4-A. 参照形式の場合
   4-A-1. 参照パス（例: refs/heads/main）を抽出
   4-A-2. .git/{参照パス}ファイルの存在確認
   4-A-3. ファイルが存在すればその内容を読み込み
   4-A-4. ファイルが存在しなければpacked-refsから検索
   4-A-5. (コミットハッシュ, false)を返却
4-B. detached形式の場合
   4-B-1. HEADの内容自体がコミットハッシュ
   4-B-2. (コミットハッシュ, true)を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[.git/HEAD読み込み]
    B --> C{HEAD内容が ref: で始まる?}
    C -->|Yes| D[参照パスを抽出]
    C -->|No| E[HEADの内容がコミットハッシュ]
    D --> F[.git/{参照パス}ファイル存在確認]
    F --> G{ファイル存在?}
    G -->|Yes| H[ファイル内容を読み込み]
    G -->|No| I[get_git_ref_entry呼び出し]
    I --> J[packed-refsから検索]
    H --> K[コミットハッシュ, false を返却]
    J --> K
    E --> L[コミットハッシュ, true を返却]
    K --> M[終了]
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-43-01 | 参照解決 | HEADが参照形式の場合、参照先のコミットハッシュを取得 | ref: で始まる場合 |
| BR-43-02 | detached判定 | HEADが直接コミットハッシュの場合はdetached=true | ref: で始まらない場合 |
| BR-43-03 | packed-refs対応 | 参照ファイルが存在しない場合はpacked-refsを検索 | loose ref不存在時 |
| BR-43-04 | ハッシュ形式 | コミットハッシュは40文字のSHA-1ハッシュ | 常時 |

### 計算ロジック

```rust
// HEAD参照解決のロジック
if head_str.starts_with("ref: ") {
    let entry = head_str["ref: ".len()..].trim_end();  // 参照パス抽出
    let git_ref = dir.join(".git").join(entry);
    let ref_str = if git_ref.is_file() {
        fs::read_to_string(&git_ref)?  // loose ref
    } else {
        get_git_ref_entry(dir, entry)?  // packed-refs検索
    };
    Ok((ref_str.trim().to_string(), false))
} else {
    Ok((head_str.trim().to_string(), true))  // detached HEAD
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-43-01 | ファイル不在 | .git/HEADファイルが存在しない | エラーメッセージを返却 |
| E-43-02 | 読み取りエラー | ファイル読み取りに失敗 | エラーメッセージを返却 |
| E-43-03 | 参照解決失敗 | loose refもpacked-refsも見つからない | エラーメッセージを返却 |

### リトライ仕様

リトライは行わない。Gitリポジトリの状態が不正な場合はエラーを返却。

## トランザクション仕様

該当なし

## パフォーマンス要件

- ファイル読み込みは軽量な操作
- packed-refs検索時は線形スキャンだが、通常は小さいファイル

## セキュリティ考慮事項

- ローカルファイルシステムのみにアクセス
- Gitリポジトリの整合性に依存

## 備考

- Git内部構造（.git/HEAD、.git/refs/、.git/packed-refs）を直接参照
- libgit2等のライブラリは使用せず、軽量な実装
- get_git_tag_rev関数でタグからのコミット解決も可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Gitの内部構造を理解する

Gitの.gitディレクトリ構造の理解が前提となる。

| 項目 | パス | 説明 |
|-----|------|------|
| HEAD | .git/HEAD | 現在のブランチ参照またはコミットハッシュ |
| loose refs | .git/refs/heads/{branch} | ブランチごとのコミットハッシュファイル |
| packed-refs | .git/packed-refs | 圧縮された参照一覧 |

**読解のコツ**: .git/HEADの内容は2パターン
- `ref: refs/heads/main` - ブランチ参照
- `a1b2c3d4...`（40文字）- detached HEAD

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fs.rs | `src/cook/fs.rs` | get_git_head_rev関数（291-308行目）- メイン処理 |

**主要処理フロー**:
1. **292行目**: .git/HEADパスの構築
2. **293-294行目**: HEADファイルの読み込み
3. **295行目**: "ref: "で始まるかの判定
4. **296行目**: 参照パスの抽出（"ref: "以降）
5. **297行目**: .git/{参照パス}のパス構築
6. **298-303行目**: loose refまたはpacked-refsからの取得
7. **304行目**: (ハッシュ, false)を返却（ブランチ参照の場合）
8. **306行目**: (ハッシュ, true)を返却（detachedの場合）

#### Step 3: packed-refs検索を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fs.rs | `src/cook/fs.rs` | get_git_ref_entry関数（317-333行目）- packed-refs検索 |

**主要処理フロー**:
- **318行目**: .git/packed-refsパスの構築
- **319-320行目**: packed-refsファイルの読み込み
- **321-328行目**: 各行をスキャンして参照名にマッチする行を検索
- **323-327行目**: マッチした行からSHAを抽出

#### Step 4: 関連ユーティリティを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | fs.rs | `src/cook/fs.rs` | get_git_tag_rev関数（311-316行目）- タグからコミット取得 |

**主要処理フロー**:
- **312行目**: 40文字の16進数ならそのまま返却（既にコミットハッシュ）
- **315行目**: タグ名からget_git_ref_entryで検索

### プログラム呼び出し階層図

```
fetch.rs / ident.rs
    │
    └─ get_git_head_rev(dir)
           ├─ fs::read_to_string(".git/HEAD")
           │
           ├─ [参照形式の場合]
           │      ├─ fs::read_to_string(".git/refs/heads/{branch}")
           │      │
           │      └─ [ファイル不在の場合]
           │             └─ get_git_ref_entry(dir, entry)
           │                    └─ fs::read_to_string(".git/packed-refs")
           │
           └─ [detachedの場合]
                  └─ HEADの内容をそのまま返却
```

### データフロー図

```
[入力]                      [処理]                           [出力]

ディレクトリパス ────────▶ .git/HEAD読み込み
                                  │
                                  ▼
                          HEAD形式判定
                                  │
              ┌───────────────────┴───────────────────┐
              ▼                                       ▼
      [参照形式]                               [detached形式]
              │                                       │
              ▼                                       │
    .git/refs/heads/{branch}                          │
       or packed-refs                                 │
              │                                       │
              └───────────────┬───────────────────────┘
                              ▼
                      (String, bool)
                      コミットハッシュ, detached状態
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fs.rs | `src/cook/fs.rs` | ソース | get_git_head_rev, get_git_ref_entry, get_git_tag_revの実装 |
| fetch.rs | `src/cook/fetch.rs` | ソース | ソースフェッチ時のHEAD取得呼び出し元 |
| ident.rs | `src/cook/ident.rs` | ソース | ビルド識別子管理でのHEAD取得呼び出し元 |
| .git/HEAD | ソースリポジトリ | Git内部 | 現在のHEAD参照 |
| .git/refs/heads/* | ソースリポジトリ | Git内部 | ブランチごとのコミットハッシュ |
| .git/packed-refs | ソースリポジトリ | Git内部 | 圧縮された参照一覧 |
